package easik.sketch.attribute;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.Arrays;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextField;

import easik.Easik;
import easik.sketch.datatype.DataType;


/**
 * Class extending JDialog which prompts the user for attribute information
 * 
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @since 2006-06-05 Kevin Green
 * @version 2006-08-21 Kevin Green
 */
public class AttributeUI extends JDialog {
	
	/**
	 * The name of the attribute
	 */
	private String _attName;
	/**
	 * The type of teh attribute
	 */
	private DataType _attType;
	/**
	 * Stores whether the attribute is new or previously created
	 */
	private boolean _isNew;
	/**
	 * Records whether the user selects to exit the attribute UI
	 */
	private boolean _exit = false;
	
	/**
	 * The JTextField used to record the name of the attribute
	 */
	private JTextField _attNameField;
	/**
	 * The JComboBox used to allow the user to select the attribute type
	 */
	private JComboBox _attTypeField;
	
	/**
	 * The UI ok button
	 */
	private JButton _okButton;
	/**
	 * The UI cancel button
	 */
	private JButton _cancelButton;
	
	/**
	 * The length of the window
	 */
	private static final int WINDOW_LENGTH = 400;
	/**
	 * The height of the window
	 */
	private static final int WINDOW_HEIGHT = 200;
	/**
	 * The window's X-position
	 */
	private static final int WINDOW_POS_X = 300;
	/**
	 * The window's Y-position
	 */
	private static final int WINDOW_POS_Y = 250;
	
	/**
	 * Creates a dialog box prompting the user for input.
	 * Used for creation of new attribute.
	 * 
	 */
	public AttributeUI(){
		super();
		this.setSize(WINDOW_LENGTH, WINDOW_HEIGHT);
		this.setResizable(false);
		this.setTitle("Attribute Values");
		this.setBounds(WINDOW_POS_X, WINDOW_POS_Y, WINDOW_LENGTH, WINDOW_HEIGHT);
		
		_isNew = true;
		
		createDialogBox();
		
		this.setModal(true);
		this.setVisible(true);
	}
	
	/**
	 * Creates a dialog box prompting the user for input.
	 * Used to edit a current attribute.
	 * 
	 * @param inAtt the attribute to be used for the UI
	 */
	public AttributeUI(EntityAttribute inAtt){
		super();
		this.setSize(WINDOW_LENGTH, WINDOW_HEIGHT);
		this.setResizable(false);
		this.setTitle("Attribute Values");
		this.setBounds(WINDOW_POS_X, WINDOW_POS_Y, WINDOW_LENGTH, WINDOW_HEIGHT);
		
		_attName = inAtt.getName();
		_attType = inAtt.getDataType();
		_isNew = false;
		
		createDialogBox();
		
		this.setModal(true);
		this.setVisible(true);
	}
	
	/**
	 * Creates a dialog box which prompts the user for attribute information
	 * 
	 */
	private void createDialogBox(){
		
		JPanel mainPane = new JPanel(new BorderLayout());
		
		//Add the title
		JLabel title = new JLabel("     ", JLabel.CENTER);
		mainPane.add(title, BorderLayout.NORTH);
		
		//Create info pane
		JPanel attInfoPane = new JPanel(new GridLayout(5, 1));
		
		//Add the attribute name text box
		attInfoPane.add(new JLabel("Attribute Name:"));
		_attNameField = new JTextField();
		_attNameField.setText(_attName);
		attInfoPane.add(_attNameField);
		
		//Add the attribute type text box
		attInfoPane.add(new JLabel("Attribute Type"));
		Object[] typeArr = Easik.getInstance().getFrame().getSketch().getDataTypeController().getDataTypes().toArray();
		Arrays.sort(typeArr, new DataTypeSorter());
		_attTypeField = new JComboBox(typeArr);
		int type = 0;
		for(int i=0; i<typeArr.length; i++){
			if(typeArr[i].equals(_attType))
				type = i;
		}
		_attTypeField.setSelectedItem(typeArr[type]);
		attInfoPane.add(_attTypeField);
		
		//Add info pane to main pane
		mainPane.add(attInfoPane, BorderLayout.CENTER);
		
		//Add the buttons
		if(_isNew){
			_okButton = new JButton("Insert");
		}
		else{
			_okButton = new JButton("Update");
		}
		_okButton.setSelected(true);
		_cancelButton = new JButton("Cancel");
		
		JPanel buttons = new JPanel(new GridLayout(1, 2));
		buttons.add(_okButton);
		buttons.add(_cancelButton);
		
		mainPane.add(buttons, BorderLayout.SOUTH);
		mainPane.setVisible(true);
		
		//Padding to make main pane look better
		mainPane.add(new JLabel("              "), BorderLayout.WEST);
		mainPane.add(new JLabel("              "), BorderLayout.EAST);
		
		this.getContentPane().add(mainPane);
		
		//Create action listeners for buttons
		_okButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				okPressed();
			}
		});
		
		_cancelButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e1) {
				cancelPressed();
			}
		});
	}
	
	/**
	 * Method called when the ok button is pressed.
	 *
	 */
	public void okPressed(){
		_attName = _attNameField.getText();
		_attType = (DataType) _attTypeField.getSelectedItem();
		if(_attName.equals("")){
			JOptionPane.showMessageDialog(Easik.getInstance().getFrame(),
					"You have entered an empty attribute name.\nPlease try again.",
					"Empty Attribute Name", JOptionPane.ERROR_MESSAGE);
			return;
		}
		_exit = true;
		dispose();
	}
	
	/**
	 * Method called when the cancel button is pressed.
	 *
	 */
	public void cancelPressed(){
		dispose();
	}
	
	/**
	 * Returns the attribute name
	 * 
	 * @return The attribute name
	 */
	public String getAttName(){
		return _attName;
	}
	
	/**
	 * Returns the attribute data type
	 * 
	 * @return The attirbute data type
	 */
	public DataType getAttType(){
		return _attType;
	}
	
	/**
	 * Returns the exit value
	 * 
	 * @return The exit value
	 */
	public boolean getExit(){
		return _exit;
	}
}